<?php
// hubtel_webhook.php
// Idempotent Hubtel MoMo webhook handler for SPA_GIG
// Requires config.php to provide:
//  - $conn (mysqli, InnoDB recommended)
//  - send_sms($to, $message) [optional]
//  - define('HUBTEL_SECRET', '...') [optional to verify signature]

include __DIR__ . '/../config.php'; // adjust this path if your config is elsewhere
date_default_timezone_set('Africa/Accra');
header('Content-Type: application/json');

// Read raw body
$raw = file_get_contents('php://input');
if (empty($raw)) {
    http_response_code(400);
    echo json_encode(['error' => 'Empty payload']);
    exit;
}

// OPTIONAL: change header name/algorithm to Hubtel docs
$signatureHeaderName = 'HTTP_X_HUBTEL_SIGNATURE'; // incoming header key in $_SERVER (e.g., 'X-Hubtel-Signature')
$incomingSignature = $_SERVER[$signatureHeaderName] ?? null;

// If HUBTEL_SECRET defined, verify HMAC SHA256 signature
if (defined('HUBTEL_SECRET') && HUBTEL_SECRET) {
    if (!$incomingSignature) {
        http_response_code(403);
        echo json_encode(['error' => 'Missing signature header']);
        exit;
    }
    $computed = hash_hmac('sha256', $raw, HUBTEL_SECRET);
    if (!hash_equals($computed, $incomingSignature)) {
        http_response_code(403);
        echo json_encode(['error' => 'Invalid signature']);
        exit;
    }
}

// Parse JSON
$payload = json_decode($raw, true);
if (!is_array($payload)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid JSON payload']);
    exit;
}

// Normalize fields (adjust to Hubtel actual keys if different)
$hubTxId   = trim($payload['transactionId'] ?? $payload['transaction_id'] ?? ($payload['hub_tx_id'] ?? ''));
$clientRef = trim($payload['clientReference'] ?? $payload['client_reference'] ?? $payload['clientRef'] ?? '');
$amount    = floatval(str_replace(',', '', ($payload['amount'] ?? $payload['Amount'] ?? 0)));
$currency  = strtoupper(trim($payload['currency'] ?? ($payload['Currency'] ?? '')));
$phone     = preg_replace('/\D+/', '', ($payload['phone'] ?? $payload['msisdn'] ?? $payload['phoneNumber'] ?? ''));
$statusRaw = strtolower(trim($payload['status'] ?? ($payload['Status'] ?? '')));

// Accept only success-like statuses
$accepted_statuses = ['success','ok','completed','completed_success'];
if (!in_array($statusRaw, $accepted_statuses)) {
    // log and ignore non-success statuses (optionally return 200 to stop retries)
    error_log("Hubtel webhook: Ignored non-success status '{$statusRaw}' for ref={$clientRef} hubTx={$hubTxId}");
    http_response_code(200);
    echo json_encode(['status'=>'ignored','reason'=>'non-success status']);
    exit;
}

// Validate basic identifiers
if (empty($hubTxId) && empty($clientRef)) {
    http_response_code(400);
    echo json_encode(['error' => 'Missing hub_tx_id and client_ref']);
    exit;
}

if ($amount <= 0) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid amount']);
    exit;
}

// DB connection check
if (!isset($conn) || !($conn instanceof mysqli)) {
    error_log("Hubtel webhook: no DB connection");
    http_response_code(500);
    echo json_encode(['error'=>'server misconfigured']);
    exit;
}

$conn->set_charset('utf8mb4');

function respond_ok($data = []) {
    http_response_code(200);
    echo json_encode(array_merge(['status'=>'ok'], $data));
    exit;
}

function respond_err($code, $msg) {
    http_response_code($code);
    echo json_encode(['error'=>$msg]);
    exit;
}

try {
    // start transaction
    $conn->begin_transaction();

    // 1) Idempotency: if hub_tx_id already exists on payments (unique index), skip
    if (!empty($hubTxId)) {
        $qi = "SELECT id, status FROM payments WHERE hub_tx_id = ? LIMIT 1 FOR UPDATE";
        $sti = $conn->prepare($qi);
        if ($sti) {
            $sti->bind_param("s", $hubTxId);
            $sti->execute();
            $resi = $sti->get_result();
            if ($resi && $resi->num_rows > 0) {
                $rowi = $resi->fetch_assoc();
                $sti->close();
                // already processed or assigned - return success to avoid replay
                $conn->commit();
                respond_ok(['skipped'=>true,'reason'=>'hub_tx_id already processed','payment_id'=>$rowi['id']]);
            }
            $sti->close();
        }
    }

    // 2) Try to find matching pending payment by clientRef (transaction_id)
    $payment = null;
    if (!empty($clientRef)) {
        $q = "SELECT id, user_id, user_phone, amount, status, transaction_id FROM payments WHERE transaction_id = ? LIMIT 1 FOR UPDATE";
        $st = $conn->prepare($q);
        if ($st) {
            $st->bind_param("s", $clientRef);
            $st->execute();
            $r = $st->get_result();
            if ($r && $r->num_rows > 0) $payment = $r->fetch_assoc();
            $st->close();
        } else {
            throw new Exception("DB prepare failed (payments lookup by transaction_id): " . $conn->error);
        }
    }

    // 3) If not found by clientRef, try to match by user_phone + amount + recent pending
    if (!$payment && !empty($phone)) {
        $q2 = "SELECT id, user_id, user_phone, amount, status, transaction_id FROM payments 
               WHERE user_phone = ? AND amount = ? AND status = 'pending' 
               ORDER BY payment_date DESC LIMIT 1 FOR UPDATE";
        $st2 = $conn->prepare($q2);
        if ($st2) {
            $st2->bind_param("sd", $phone, $amount);
            $st2->execute();
            $r2 = $st2->get_result();
            if ($r2 && $r2->num_rows > 0) $payment = $r2->fetch_assoc();
            $st2->close();
        }
    }

    // 4) If still not found, record webhook in processed_webhooks for traceability and exit
    if (!$payment) {
        $note = "No matching pending payment found (clientRef={$clientRef} hubTx={$hubTxId} phone={$phone} amount={$amount})";
        $ins = $conn->prepare("INSERT INTO processed_webhooks (hub_tx_id, client_ref, payment_id, raw_payload, created_at, note) VALUES (?, ?, NULL, ?, NOW(), ?)");
        if ($ins) {
            $ins->bind_param("ssss", $hubTxId, $clientRef, $raw, $note);
            $ins->execute();
            $ins->close();
        }
        $conn->commit();
        respond_ok(['skipped'=>true,'reason'=>'no matching payment']);
    }

    // 5) If payment.status is already success, skip (idempotent)
    if (strtolower($payment['status']) === 'success' || strtolower($payment['status']) === 'completed') {
        // still record processed_webhooks
        $ins2 = $conn->prepare("INSERT INTO processed_webhooks (hub_tx_id, client_ref, payment_id, raw_payload, created_at, note) VALUES (?, ?, ?, ?, NOW(), ?)");
        if ($ins2) {
            $note2 = "Payment already success";
            $pid = intval($payment['id']);
            $ins2->bind_param("ssiss", $hubTxId, $clientRef, $pid, $raw, $note2);
            $ins2->execute();
            $ins2->close();
        }
        $conn->commit();
        respond_ok(['skipped'=>true,'reason'=>'payment already success','payment_id'=>$payment['id']]);
    }

    // 6) Amount sanity-check: prevent tampering
    $dbAmount = floatval($payment['amount']);
    if (abs($dbAmount - $amount) > 0.5) {
        // significant mismatch: rollback and alert
        throw new Exception("Amount mismatch: DB={$dbAmount} webhook={$amount}");
    }

    // 7) Update payments row: set hub_tx_id, status='success', confirmed_at, webhook_status='processed'
    $upd = $conn->prepare("UPDATE payments SET status = 'success', hub_tx_id = ?, confirmed_at = NOW(), webhook_status = 'processed' WHERE id = ?");
    if (!$upd) throw new Exception("DB prepare failed (update payments): " . $conn->error);
    $pid = intval($payment['id']);
    $upd->bind_param("si", $hubTxId, $pid);
    if (!$upd->execute()) throw new Exception("Failed updating payments row: " . $upd->error);
    $upd->close();

    // 8) Lock user row and update balance atomically
    $uid = intval($payment['user_id']);
    if ($uid <= 0) throw new Exception("Payment has no user_id (payment id {$pid})");

    // select user for update
    $su = $conn->prepare("SELECT balance, phone, full_name FROM users WHERE id = ? LIMIT 1 FOR UPDATE");
    if (!$su) throw new Exception("DB prepare failed (select user): " . $conn->error);
    $su->bind_param("i", $uid);
    $su->execute();
    $ur = $su->get_result()->fetch_assoc();
    $su->close();
    if (!$ur) throw new Exception("User not found for id {$uid}");

    $oldBalance = floatval($ur['balance'] ?? 0.0);
    $newBalance = $oldBalance + $amount;

    $uu = $conn->prepare("UPDATE users SET balance = ? WHERE id = ?");
    if (!$uu) throw new Exception("DB prepare failed (update user): " . $conn->error);
    $uu->bind_param("di", $newBalance, $uid);
    if (!$uu->execute()) throw new Exception("Failed updating user balance: " . $uu->error);
    $uu->close();

    // 9) Insert into payment_events ledger (create this table if you haven't yet)
    $evt = $conn->prepare("INSERT INTO payment_events (payment_id, user_id, user_phone, hub_tx_id, amount, old_balance, new_balance, event_time, raw_payload) VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), ?)");
    if (!$evt) throw new Exception("DB prepare failed (insert event): " . $conn->error);
    $userPhone = $payment['user_phone'] ?? $phone;
    $evt->bind_param("iissddds", $pid, $uid, $userPhone, $hubTxId, $amount, $oldBalance, $newBalance, $raw);
    // NOTE: adjust bind types above if your payment_events schema differs
    if (!$evt->execute()) throw new Exception("Failed inserting payment event: " . $evt->error);
    $evt->close();

    // 10) Record processed_webhooks
    $insPH = $conn->prepare("INSERT INTO processed_webhooks (hub_tx_id, client_ref, payment_id, raw_payload, created_at, note) VALUES (?, ?, ?, ?, NOW(), ?)");
    if ($insPH) {
        $note = "processed";
        $insPH->bind_param("ssiss", $hubTxId, $clientRef, $pid, $raw, $note);
        $insPH->execute();
        $insPH->close();
    }

    // commit
    $conn->commit();

    // 11) Send SMS receipt (best-effort)
    $userPhoneToSend = $ur['phone'] ?? $userPhone;
    $userName = $ur['full_name'] ?? '';
    $smsMsg = "Payment received: GHS " . number_format($amount,2) . ". Ref: {$clientRef}. New balance: GHS " . number_format($newBalance,2);
    if (function_exists('send_sms')) {
        try { @send_sms($userPhoneToSend, $smsMsg); } catch (Exception $e) { error_log("SMS send failed: " . $e->getMessage()); }
    }

    respond_ok(['processed'=>true,'payment_id'=>$pid]);

} catch (Exception $e) {
    // rollback and return 500
    if ($conn->in_transaction) $conn->rollback();
    error_log("Hubtel webhook processing error: " . $e->getMessage());
    respond_err(500, 'processing_error');
}
