<?php
include('../config.php');
session_start();

$user_id = $_SESSION['user_id'] ?? null;
if (!$user_id) { 
    header("Location: login.php"); 
    exit; 
}

// Fetch user details
$stmt = $conn->prepare("SELECT id, full_name, phone, balance FROM users WHERE id=? LIMIT 1");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$user = $stmt->get_result()->fetch_assoc();
$stmt->close();

// Fetch recent contributions
$stmt = $conn->prepare("SELECT amount, status, payment_date FROM payments WHERE user_id=? ORDER BY payment_date DESC LIMIT 5");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$contributions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Fetch chart data
$chartLabels = [];
$chartData = [];

$stmt = $conn->prepare("
SELECT DATE_FORMAT(payment_date, '%Y-%m') as month, SUM(amount) as total
FROM payments
WHERE user_id = ? AND status = 'success'
GROUP BY month
ORDER BY month ASC
");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();

while ($row = $result->fetch_assoc()) {
    $chartLabels[] = date('M Y', strtotime($row['month']));
    $chartData[] = (float)$row['total'];
}
$stmt->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>3y3 SUPA Dashboard</title>

<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

<style>

/* ===== GLOBAL ===== */
body {
    margin: 0;
    font-family: 'Poppins', sans-serif;
    background: #F4F6F9;
    display: flex;
}

/* ===== SIDEBAR ===== */
.sidebar {
    width: 240px;
    background: #002147;
    color: #fff;
    position: fixed;
    left: 0;
    top: 0;
    bottom: 0;
    display: flex;
    flex-direction: column;
    padding: 20px 0;
}

.sidebar .logo {
    text-align: center;
    margin-bottom: 25px;
}

.sidebar a {
    padding: 14px 25px;
    text-decoration: none;
    color: #fff;
    display: flex;
    align-items: center;
    gap: 12px;
    font-size: 14px;
    transition: 0.3s;
}

.sidebar a:hover,
.sidebar a.active {
    background: #C9A227;
    color: #002147;
    font-weight: 600;
}

/* Push logout to bottom */
.sidebar a.logout {
    margin-top: auto;
}

/* ===== MAIN ===== */
.main {
    margin-left: 240px;
    width: 100%;
    min-height: 100vh;
    display: flex;
    flex-direction: column;
}

/* HEADER */
header {
    background: #fff;
    padding: 15px 30px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    box-shadow: 0 3px 10px rgba(0,0,0,0.05);
}

/* NATIONAL BAR */
.national-bar {
    background: #C9A227;
    color: #002147;
    padding: 8px 30px;
    font-size: 13px;
    font-weight: 600;
    display: flex;
    justify-content: space-between;
}

/* CONTENT CONTAINER (FIXED WIDTH) */
.content {
    max-width: 1100px;
    margin: 30px auto;
    padding: 0 20px;
    width: 100%;
}

/* EXECUTIVE PANEL */
.executive-panel {
    background: #fff;
    border-radius: 14px;
    padding: 20px 25px;
    margin-bottom: 25px;
    display: flex;
    justify-content: space-between;
    box-shadow: 0 8px 25px rgba(0,0,0,0.06);
}

.exec-item small {
    color: #777;
    font-size: 12px;
}

.exec-item h4 {
    margin: 5px 0 0;
    color: #002147;
}

.active-status {
    color: #1E7E34;
}

/* ACTION CARDS */
.cards {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 18px;
    margin-bottom: 30px;
}

.card {
    background: #fff;
    border-radius: 14px;
    padding: 18px;
    text-align: center;
    border: 1px solid #eee;
    box-shadow: 0 6px 18px rgba(0,0,0,0.05);
    transition: 0.2s ease;
    cursor: pointer;
    font-size: 14px;
}

.card:hover {
    transform: translateY(-3px);
}

.card i {
    font-size: 20px;
    color: #C9A227;
    margin-bottom: 6px;
}

/* CHART */
.chart-card {
    background: #fff;
    padding: 20px;
    border-radius: 14px;
    box-shadow: 0 8px 25px rgba(0,0,0,0.06);
    margin-bottom: 30px;
    max-width: 750px;
    margin-left: auto;
    margin-right: auto;
}

.chart-wrapper {
    height: 280px;
}

/* CONTRIBUTIONS */
.contribution-list {
    list-style: none;
    padding: 0;
    max-width: 750px;
    margin: 0 auto;
}

.contribution-list li {
    background: #fff;
    margin-bottom: 12px;
    padding: 14px 18px;
    border-radius: 12px;
    display: flex;
    justify-content: space-between;
    box-shadow: 0 4px 15px rgba(0,0,0,0.05);
}

.status {
    padding: 5px 12px;
    border-radius: 20px;
    font-size: 11px;
    font-weight: 600;
}

.status.success { background: #E6F4EA; color: #1E7E34; }
.status.pending { background: #FFF8E1; color: #856404; }
.status.failed { background: #FDECEA; color: #721C24; }

/* MOBILE */
@media(max-width: 900px){
    .sidebar {
        width: 70px;
    }

    .sidebar a span {
        display: none;
    }

    .main {
        margin-left: 70px;
    }

    .cards {
        grid-template-columns: 1fr;
    }
}

</style>
</head>

<body>

<div class="sidebar">
    <div class="logo">
        <div style="font-weight:600;">3y3 SUPA</div>
    </div>

    <a href="dashboard.php" class="active"><i class="fa-solid fa-gauge"></i><span>Dashboard</span></a>
    <a href="contribute.php"><i class="fa-solid fa-money-bill-wave"></i><span>Contribute</span></a>
    <a href="history.php"><i class="fa-solid fa-clock-rotate-left"></i><span>History</span></a>
    <a href="projection.php"><i class="fa-solid fa-chart-line"></i><span>Projection</span></a>
    <a href="support.php"><i class="fa-solid fa-headset"></i><span>Support</span></a>
    <a href="beneficiaries.php"><i class="fa-solid fa-users"></i><span>Beneficiaries</span></a>
    <a href="balance.php"><i class="fa-solid fa-wallet"></i><span>Balance</span></a>

    <a href="logout.php" class="logout">
        <i class="fa-solid fa-right-from-bracket"></i><span>Logout</span>
    </a>
</div>

<div class="main">

<header>
    <div><i class="fa-solid fa-user"></i> <?=htmlspecialchars($user['full_name'])?></div>
</header>

<div class="national-bar">
    <div>3y3 Supa</div>
    <div>Secure Today. Enjoy Tomorrow.</div>
</div>

<div class="content">

    <div class="executive-panel">
        <div class="exec-item">
            <small>Member ID</small>
            <h4>3YS-<?=str_pad($user['id'],6,'0',STR_PAD_LEFT)?></h4>
        </div>

        <div class="exec-item">
            <small>Total Contributions</small>
            <h4>₵ <?=number_format($user['balance'],2)?></h4>
        </div>

        <div class="exec-item">
            <small>Status</small>
            <h4 class="active-status">Active</h4>
        </div>
    </div>

    <div class="cards">
        <div class="card" onclick="window.location='contribute.php'"><i class="fa-solid fa-money-bill-wave"></i><div>Contribute</div></div>
        <div class="card" onclick="window.location='history.php'"><i class="fa-solid fa-clock-rotate-left"></i><div>History</div></div>
        <div class="card" onclick="window.location='projection.php'"><i class="fa-solid fa-chart-line"></i><div>Projection</div></div>
        <div class="card" onclick="window.location='support.php'"><i class="fa-solid fa-headset"></i><div>Support</div></div>
        <div class="card" onclick="window.location='beneficiaries.php'"><i class="fa-solid fa-users"></i><div>Beneficiaries</div></div>
        <div class="card" onclick="window.location='balance.php'"><i class="fa-solid fa-wallet"></i><div>Balance</div></div>
    </div>

    <div class="chart-card">
        <h3>Pension Growth</h3>
        <div class="chart-wrapper">
            <canvas id="growthChart"></canvas>
        </div>
    </div>

    <h3 style="text-align:center;">Recent Contributions</h3>
    <ul class="contribution-list">
        <?php foreach($contributions as $c): ?>
            <li>
                <div>
                    <strong>₵ <?=number_format($c['amount'],2)?></strong><br>
                    <small><?=date('d M Y', strtotime($c['payment_date']))?></small>
                </div>
                <span class="status <?=strtolower($c['status'])?>">
                    <?=ucfirst($c['status'])?>
                </span>
            </li>
        <?php endforeach; ?>
    </ul>

</div>
</div>

<script>
const ctx = document.getElementById('growthChart');

new Chart(ctx, {
    type: 'line',
    data: {
        labels: <?= json_encode($chartLabels) ?>,
        datasets: [{
            data: <?= json_encode($chartData) ?>,
            borderColor: '#002147',
            backgroundColor: 'rgba(201,162,39,0.2)',
            fill: true,
            tension: 0.3
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: { legend: { display: false } },
        scales: { y: { beginAtZero: true } }
    }
});
</script>

</body>
</html>