<?php
// contribute.php
include('../config.php');
session_start();

$user_id = $_SESSION['user_id'] ?? null;
if (!$user_id) {
    header("Location: login.php");
    exit;
}

// Fetch user info
$stmt = $conn->prepare("SELECT full_name, balance, phone, sector FROM users WHERE id=? LIMIT 1");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$user = $stmt->get_result()->fetch_assoc();
$stmt->close();

// Handle form submission
$success = $error = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $amount = floatval($_POST['amount'] ?? 0);

    if ($amount <= 0) {
        $error = "Please enter a valid amount.";
    } else {
        // Fetch sector minimum
        $stmt = $conn->prepare("SELECT minimum_amount FROM sector_minimums WHERE sector_name=? LIMIT 1");
        $stmt->bind_param("s", $user['sector']);
        $stmt->execute();
        $row = $stmt->get_result()->fetch_assoc();
        $stmt->close();

        $month_min = floatval($row['minimum_amount'] ?? 50);
        if ($month_min < 1) $month_min = 50;

        // Calculate months covered
        $monthsCovered = floor($amount / $month_min);
        if ($monthsCovered < 1) $monthsCovered = 1;

        // Generate transaction reference
        $clientRef = strtoupper(bin2hex(random_bytes(6)));
        $created_at = date("Y-m-d H:i:s");

        // Insert into payments table
        $stmt = $conn->prepare("INSERT INTO payments (user_id, user_phone, amount, transaction_id, status, payment_date) VALUES (?, ?, ?, ?, 'pending', ?)");
        $stmt->bind_param("isdss", $user_id, $user['phone'], $amount, $clientRef, $created_at);

        if ($stmt->execute()) {
            $stmt->close();

            // Log API request
            $req = json_encode([
                "ClientReference" => $clientRef,
                "Amount" => $amount,
                "Phone" => $user['phone'],
                "MonthsCovered" => $monthsCovered
            ], JSON_UNESCAPED_SLASHES);

            $type = 'PAYMENT_REQUEST';
            $stmt2 = $conn->prepare("INSERT INTO api_logs (response, token, date_created, type) VALUES (?, ?, ?, ?)");
            $stmt2->bind_param("ssss", $req, $clientRef, $created_at, $type);
            $stmt2->execute();
            $stmt2->close();

            // Initiate Hubtel MoMo payment
            // include('../includes/mpay_api.php'); // make sure Hubtel API function exists
            $resp = hubtel_initiate_payment($user['phone'], $amount, $clientRef, "Contribution to EdwumaSupa");
            if (!$resp) error_log("Hubtel initiation failed for {$user['phone']}, Ref: $clientRef");

            $success = "Payment request created (Ref: $clientRef). Approve on your phone to complete.";
        } else {
            $stmt->close();
            $error = "Unable to create payment. Try again later.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>3y3 SUPA Dashboard</title>
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>


<style>

/* ===== GLOBAL ===== */
body {
    margin: 0;
    font-family: 'Poppins', sans-serif;
    background: #F4F6F9;
    display: flex;
}

/* ===== SIDEBAR ===== */
.sidebar {
    width: 240px;
    background: #002147;
    color: #fff;
    position: fixed;
    left: 0;
    top: 0;
    bottom: 0;
    display: flex;
    flex-direction: column;
    padding: 20px 0;
}

.sidebar .logo {
    text-align: center;
    margin-bottom: 25px;
}

.sidebar a {
    padding: 14px 25px;
    text-decoration: none;
    color: #fff;
    display: flex;
    align-items: center;
    gap: 12px;
    font-size: 14px;
    transition: 0.3s;
}

.sidebar a:hover,
.sidebar a.active {
    background: #C9A227;
    color: #002147;
    font-weight: 600;
}

/* Push logout to bottom */
.sidebar a.logout {
    margin-top: auto;
}

/* ===== MAIN ===== */
.main {
    margin-left: 240px;
    width: 100%;
    min-height: 100vh;
    display: flex;
    flex-direction: column;
}

/* HEADER */
header {
    background: linear-gradient(135deg, #002147 0%, #001530 100%);
    border-bottom: 4px solid #C9A227;
    padding: 15px 20px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    color: #fff;
}


/* NATIONAL BAR */
.national-bar {
    background: #C9A227;
    color: #002147;
    padding: 8px 30px;
    font-size: 13px;
    font-weight: 600;
    display: flex;
    justify-content: space-between;
}

/* CONTENT CONTAINER (FIXED WIDTH) */
.content {
    max-width: 1100px;
    margin: 30px auto;
    padding: 0 20px;
    width: 100%;
}

.logo-area {
    display: flex;
    align-items: center;
    gap: 12px;
}

.logo-area img {
    height: 45px;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.logout-link {
    color: #fff;
    font-size: 18px;
}

/* National Bar */
.national-bar {
    background: #C9A227;
    color: #002147;
    padding: 8px 20px;
    font-size: 13px;
    font-weight: 600;
    display: flex;
    justify-content: space-between;
}

/* Container */
.container {
    padding: 0px;
}

/* Executive Panel */
.executive-panel {
    background: #fff;
    border-radius: 16px;
    padding: 20px;
    margin-bottom: 25px;
    display: flex;
    justify-content: space-between;
    box-shadow: 0 10px 30px rgba(0,0,0,0.08);
}

.exec-item small {
    color: #5A5A5A;
}

.exec-item h3 {
    margin-top: 5px;
    color: #002147;
}

.active-status {
    color: #1E7E34;
}

/* Cards */
.cards {
    display: grid;
    grid-template-columns: repeat(2, 1fr); /* ALWAYS 2 per row */
    gap: 15px;
    margin-bottom: 25px;
}

.card {
    background: #fff;
    border-radius: 18px;
    padding: 16px; /* reduced padding */
    text-align: center;
    border: 1px solid #E5E5E5;
    box-shadow: 0 6px 18px rgba(0,0,0,0.05);
    transition: 0.25s ease;
    cursor: pointer;
    min-height: 120px; /* keeps them compact */
    display: flex;
    flex-direction: column;
    justify-content: center;
}

.card:hover {
    transform: translateY(-4px);
    box-shadow: 0 10px 25px rgba(0,0,0,0.1);
}

.card i {
    font-size: 24px; /* slightly smaller icon */
    color: #C9A227;
    margin-bottom: 8px;
}

.card div {
    font-weight: 600;
    color: #002147;
    font-size: 15px;
}

/* CHART */
.chart-card {
    background: #fff;
    padding: 20px;
    border-radius: 14px;
    box-shadow: 0 8px 25px rgba(0,0,0,0.06);
    margin-bottom: 30px;
    max-width: 750px;
    margin-left: auto;
    margin-right: auto;
}

.chart-wrapper {
    height: 280px;
}

/* Contributions */
.section h2 {
    color: #002147;
}

.contribution-list {
    list-style: none;
    padding: 0;
}

.contribution-list li {
    background: #fff;
    margin-bottom: 10px;
    padding: 15px;
    border-radius: 12px;
    display: flex;
    justify-content: space-between;
    box-shadow: 0 4px 15px rgba(0,0,0,0.05);
}

.status {
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 600;
}

.status.success { background: #E6F4EA; color: #1E7E34; }
.status.pending { background: #FFF8E1; color: #856404; }
.status.failed { background: #FDECEA; color: #721C24; }


footer {
    position: fixed;
    bottom: 0;
    left: 0;
    width: 100%;
    background: #f5f5f5; /* light grey / whitish */
    color: #333; /* dark text for contrast */
    text-align: center;
    padding: 12px 0;
    font-size: 14px;
    box-shadow: 0 -3px 10px rgba(0,0,0,0.1);
}

footer a {
    color: #002147; /* optional: make any links dark blue */
    text-decoration: none;
}



/* MOBILE */
@media(max-width: 900px){
    .sidebar {
        width: 70px;
    }

    .sidebar a span {
        display: none;
    }

    .main {
        margin-left: 70px;
    }

    .cards {
        grid-template-columns: 1fr;
    }
}

.header-right { display:flex; align-items:center; gap:15px; }
.header-right .logout-link { color:#fff; font-size:18px; margin-left:5px; }
.container { padding: 0px 20px 20px 20px; max-width:400px; margin:auto; }
.balance-box { background:#fff; padding:15px; border-radius:12px; box-shadow:0 6px 20px rgba(0,0,0,0.08); text-align:center; margin-bottom:20px; }
.balance-box h3 { margin:0 0 5px; color:#C9A227; }
.balance-box span { font-size:20px; font-weight:600; }
form { background:#fff; padding:20px; border-radius:12px; box-shadow:0 6px 20px rgba(0,0,0,0.08); }
form input, form button { width:100%; padding:12px; margin-bottom:15px; border-radius:10px; border:1px solid #ccc; font-size:16px; outline:none; }
form input:focus { border-color:#F58220; box-shadow:0 0 6px rgba(245,130,32,0.3); }
form button { background: linear-gradient(135deg, #F58220, #C9A227); color:#fff; font-weight:600; border:none; cursor:pointer; transition:0.3s; }
form button:hover { background: linear-gradient(135deg, #D85C00, #F58220); }
.message { text-align:center; margin-bottom:15px; font-weight:500; }
.success { color:#28a745; }
.error { color:#dc3545; }

input {
    width: 80%;
    padding: 12px 15px;
    margin-bottom: 12px;
    border-radius: 8px;
    border: 1px solid #ccc;
    font-size: 16px;
    box-sizing: border-box;
}

</style>
<body>

<div class="sidebar">
    <div class="logo">
        <div style="font-weight:600;">3y3 SUPA</div>
    </div>

    <a href="dashboard.php"><i class="fa-solid fa-gauge"></i><span>Dashboard</span></a>
    <a href="contribute.php" class="active"><i class="fa-solid fa-money-bill-wave"></i><span>Contribute</span></a>
    <a href="history.php"><i class="fa-solid fa-clock-rotate-left"></i><span>History</span></a>
    <a href="projection.php"><i class="fa-solid fa-chart-line"></i><span>Projection</span></a>
    <a href="support.php"><i class="fa-solid fa-headset"></i><span>Support</span></a>
    <a href="beneficiaries.php"><i class="fa-solid fa-users"></i><span>Beneficiaries</span></a>
    <a href="balance.php"><i class="fa-solid fa-wallet"></i><span>Balance</span></a>

    <a href="logout.php" class="">
        <i class="fa-solid fa-right-from-bracket"></i><span>Logout</span>
    </a>
</div>

<div class="main">

<header>
    <div><i class="fa-solid fa-user"></i> <?=htmlspecialchars($user['full_name'])?></div>
</header>

<div class="national-bar">
    <div>3y3 Supa</div>
    <div>Secure Today. Enjoy Tomorrow.</div>
</div>

<div class="container">

    <div class="balance-box">
        <h3>Current Balance</h3>
        <span>₵ <?=number_format($user['balance'],2)?></span>
    </div>

    <?php if($success): ?>
        <div class="message success"><?=htmlspecialchars($success)?></div>
    <?php endif; ?>
    <?php if($error): ?>
        <div class="message error"><?=htmlspecialchars($error)?></div>
    <?php endif; ?>

    <form method="POST" action="">
        <input type="number" step="0.01" name="amount" placeholder="Enter contribution amount (₵)" required>
        <button type="submit">Contribute</button>
    </form>

</div>
</div>
<footer>
    &copy; <?=date('Y')?> 3y3 Supa. Powered by Tekpulse Consult Ltd.
</footer>

</body>
</html>
