<?php
// ussd_callback.php - 3y3 Supa USSD Entry Point (fixed & hardened)
// Expects config.php to provide $conn (mysqli), send_sms(), hubtel_initiate_payment()
include("config.php");
date_default_timezone_set('Africa/Accra');

// Read raw input
$raw = file_get_contents('php://input');
if (empty($raw)) {
    http_response_code(400);
    exit('No input');
}

$data = json_decode($raw, true);
if (!$data) parse_str($raw, $data);
if (!is_array($data)) $data = [];

// Helpers: incoming fields
$sessionID = $data['sessionID'] ?? null;
$userID    = $data['userID'] ?? null;
$newSession= isset($data['newSession']) ? (bool)$data['newSession'] : false;
$msisdn    = preg_replace('/\D+/', '', $data['msisdn'] ?? '');
$userData  = trim($data['userData'] ?? '');
$network   = strtoupper(trim($data['network'] ?? ''));
$provider  = strtoupper(trim($data['provider'] ?? ''));

// Basic validation
if (!$sessionID || !$msisdn) {
    http_response_code(400);
    echo json_encode(["error"=>"Missing sessionID or msisdn"]);
    exit;
}

// Sessions storage
$sessionsDir = __DIR__ . "/sessions";
if (!is_dir($sessionsDir)) @mkdir($sessionsDir, 0755, true);
$sessionFile = "$sessionsDir/$sessionID.json";

// Load or init session
$session = null;
if ($newSession || !file_exists($sessionFile)) {
    $session = ['step' => 'MAIN_MENU', 'created_at' => date("Y-m-d H:i:s"), 'msisdn' => $msisdn];
} else {
    $content = @file_get_contents($sessionFile);
    $session = json_decode($content, true);
    if (!is_array($session)) $session = ['step'=>'MAIN_MENU','msisdn'=>$msisdn];
}

// Helper: save session (atomic-ish)
function save_session($path, $payload) {
    @file_put_contents($path, json_encode($payload));
}

// Helper to respond USSD (exits)
function ussd_response($sessionID, $userID, $msisdn, $message, $continueSession=true) {
    $resp = ["sessionID"=>$sessionID,"userID"=>$userID,"msisdn"=>$msisdn,"message"=>$message,"continueSession"=>$continueSession];
    header('Content-Type: application/json');
    echo json_encode($resp);
    exit;
}

// Helper sanitize numeric amounts
function sanitize_amount($s) {
    $s = str_replace(',', '', $s);
    $s = preg_replace('/[^0-9.]/', '', $s);
    return is_numeric($s) ? (float)$s : 0;
}

// Fetch user (if exists)
$user = null;
if (isset($conn) && $msisdn) {
    $stmt = $conn->prepare("
        SELECT id, full_name, phone, balance, pin, emergency_name, emergency_phone, emergency_relation, sector 
        FROM users 
        WHERE phone = ? 
        LIMIT 1
    ");
    if ($stmt) {
        $stmt->bind_param("s", $msisdn);
        $stmt->execute();
        $res = $stmt->get_result();
        if ($res) $user = $res->fetch_assoc();
        $stmt->close();
    } else {
        error_log("DB prepare failed (fetch user): " . ($conn->error ?? ''));
    }
}

// State & input
$step = $session['step'] ?? 'MAIN_MENU';
$input = $userData;

// MAIN MENU
if ($step === 'MAIN_MENU') {
    if ($user) {
        $displayName = $user['full_name'] ? mb_strtoupper($user['full_name']) : $user['phone'];
        $message = "Welcome $displayName\n1. Contribute\n2. My Balance\n3. History\n4. Pension Projection\n5. Beneficiaries\n6. Support\n7. My PIN\n8. Contribution Status\n0. Exit";
        $session['step'] = 'MENU_CHOICE';
        save_session($sessionFile, $session);
        ussd_response($sessionID, $userID, $msisdn, $message, true);
    } else {
        $message = "Welcome to 3y3 Supa\n1. Register\n2. How it works\n0. Exit";
        $session['step'] = 'GUEST_MENU';
        save_session($sessionFile, $session);
        ussd_response($sessionID, $userID, $msisdn, $message, true);
    }
}

// ---------------------------------
// GUEST MENU
// ---------------------------------
if ($step==='GUEST_MENU') {
    if ($input==="1") {
        $session['step']='REG_FNAME';
        save_session($sessionFile,$session);
        ussd_response($sessionID,$userID,$msisdn,"Enter first name:",true);
    } elseif ($input==="2") {
$msg = "3y3 Supa:\n"
     . "1. Helps gig workers & informal sector save for retirement.\n"
     . "2. Contribute anytime via MobileMoney.\n"
     . "3. Check balance, payment history & manage beneficiaries.\n"
     . "4. Start small, grow savings & secure your future!";
        ussd_response($sessionID,$userID,$msisdn,$msg,false);
    } elseif ($input==="0") ussd_response($sessionID,$userID,$msisdn,"Goodbye.",false);
    else ussd_response($sessionID,$userID,$msisdn,"Invalid option. 1.Register 2.How it works 0.Exit",true);
}

// Registration Flow
if ($step === 'REG_FNAME') {
    $session['reg_fname'] = substr(trim($input), 0, 50);
    $session['step'] = 'REG_LNAME';
    save_session($sessionFile, $session);
    ussd_response($sessionID, $userID, $msisdn, "Enter last name:", true);
}
if ($step === 'REG_LNAME') {
    $session['reg_lname'] = substr(trim($input), 0, 50);
    $session['step'] = 'REG_PIN';
    save_session($sessionFile, $session);
    ussd_response($sessionID, $userID, $msisdn, "Set a 4-digit PIN (for security):", true);
}
if ($step === 'REG_PIN') {
    $pin = preg_replace('/\D/', '', $input);
    if (strlen($pin) !== 4) {
        ussd_response($sessionID, $userID, $msisdn, "PIN must be 4 digits. Enter PIN:", true);
    }

    // Store plain PIN temporarily for SMS
    $session['reg_pin_plain'] = $pin;
    $session['reg_pin'] = password_hash($pin, PASSWORD_DEFAULT);

    $session['step'] = 'REG_CONFIRM';
    save_session($sessionFile, $session);
    ussd_response($sessionID, $userID, $msisdn, "Confirm registration?\n1. Confirm\n2. Cancel", true);
}


if ($step==='REG_CONFIRM') {
    if ($input==="1") {
        $fullname = trim(($session['reg_fname'] ?? '').' '.($session['reg_lname'] ?? ''));
        $phone = $msisdn;
        $pin_plain = $session['reg_pin_plain'];
        $pin_hash = $session['reg_pin'];

        $stmt = $conn->prepare("INSERT INTO users (full_name,phone,network,pin,balance,registration_channel,date_registered) VALUES (?,?,?,?,0,'USSD',NOW())");
        $stmt->bind_param("ssss",$fullname,$phone,$network,$pin_hash);

        if($stmt->execute()) {
            $stmt->close();

            $loginLink="https://tuliyapayinc.com/edwumasupa/mobile/login.php";
            $smsMsg="Welcome $fullname to 3y3 Supa!\nLogin with phone: $phone and PIN: $pin_plain\nPortal: $loginLink";
            send_sms($phone,$smsMsg);

            $session['step']='MAIN_MENU';
            save_session($sessionFile,$session);
            ussd_response($sessionID,$userID,$msisdn,"Registration successful. Check SMS for login details.",false);
        } else {
            $stmt->close();
            ussd_response($sessionID,$userID,$msisdn,"Registration failed. Try again later.",false);
        }
    } else {
        ussd_response($sessionID,$userID,$msisdn,"Registration cancelled.",false);
    }
}

// MENU CHOICE (Authenticated Users)
if ($step === 'MENU_CHOICE') {
    switch ($input) {

        case "1": // Contribute
            $session['step'] = 'CONTRIB_AMOUNT';
            save_session($sessionFile, $session);
            ussd_response($sessionID, $userID, $msisdn, "Enter amount to contribute (GHS):", true);
            break;

        case "2": // My Balance
            $stmt = $conn->prepare("SELECT balance FROM users WHERE phone = ? LIMIT 1");
            $stmt->bind_param("s", $msisdn);
            $stmt->execute();
            $r = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            $bal = number_format(floatval($r['balance'] ?? 0), 2);
            ussd_response($sessionID, $userID, $msisdn, "Your balance: GHS $bal", false);
            break;

        case "3": // History
            $stmt = $conn->prepare("SELECT amount, status, payment_date, transaction_id 
                                    FROM payments 
                                    WHERE user_phone=? 
                                    ORDER BY payment_date DESC LIMIT 10");
            $stmt->bind_param("s", $msisdn);
            $stmt->execute();
            $res = $stmt->get_result();
            $history = [];
            while ($row = $res->fetch_assoc()) {
                $history[] = ($row['payment_date'] ?? '') . ": GHS " . number_format(floatval($row['amount']),2) . " [" . strtoupper($row['status'] ?? '') . "]";
            }
            $stmt->close();

            $msg = !empty($history) ? "Last Payments:\n".implode("\n", $history) 
                                    : "No payment history found.";

            $session['step'] = 'MAIN_MENU';
            save_session($sessionFile, $session);

            ussd_response($sessionID, $userID, $msisdn, $msg, false);
            break;

        case "4": // Pension Projection
            $bal = floatval($user['balance'] ?? 0);
            $projection = $bal * pow(1.10, 5); // Simple 5-year projection, 10% yearly
            $msg = "Current balance: GHS ".number_format($bal,2)
                 . "\nProjected in 5 years: GHS ".number_format($projection,2);

            $session['step'] = 'MAIN_MENU';
            save_session($sessionFile, $session);

            ussd_response($sessionID, $userID, $msisdn, $msg, false);
            break;

       case "5": // Beneficiaries Main Menu
            $msg = "Beneficiaries\n1. Add Beneficiary\n2. View Saved Beneficiaries\n0. Back";
            $session['step'] = 'BEN_MAIN';
            save_session($sessionFile, $session);
            ussd_response($sessionID, $userID, $msisdn, $msg, true);
            break;

        case "6": // Support
            $session['step'] = 'SUPPORT_MESSAGE';
            save_session($sessionFile, $session);
            ussd_response($sessionID, $userID, $msisdn, "Enter your support message:", true);
            break;

        case "7": // Change PIN
            $session['step'] = 'CHANGE_PIN_OLD';
            save_session($sessionFile, $session);
            ussd_response($sessionID, $userID, $msisdn, "Enter your current 4-digit PIN:", true);
            break;

        case "8": // Contribution Status
            $month = date("Y-m");

            // Get user's sector minimum
            $sector = $user['sector'] ?? '';
            $stmt = $conn->prepare("SELECT minimum_amount FROM sector_minimums WHERE sector_name=? LIMIT 1");
            $stmt->bind_param("s", $sector);
            $stmt->execute();
            $rowMin = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            $minimum = floatval($rowMin['minimum_amount'] ?? 0);

            // Fetch monthly contribution record
            $stmt = $conn->prepare("SELECT total_paid, status FROM monthly_contributions WHERE user_id=? AND contribution_month=? LIMIT 1");
            $uid = intval($user['id'] ?? 0);
            $stmt->bind_param("is", $uid, $month);
            $stmt->execute();
            $rowC = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            if (!$rowC) {
                $msg = "No contributions recorded this month.\n"
                     . "Minimum required: GHS ".number_format($minimum,2)
                     . "\nProgress: GHS 0 / GHS ".number_format($minimum,2);
            } else {
                $paid = floatval($rowC['total_paid']);
                $status_text = strtoupper($rowC['status'] ?? 'UNKNOWN');
                $msg = "Contribution Status ($month)\n"
                     . "Paid: GHS ".number_format($paid,2)."\n"
                     . "Required: GHS ".number_format($minimum,2)."\n"
                     . "Progress: GHS ".number_format($paid,2)." / GHS ".number_format($minimum,2)."\n"
                     . "Status: $status_text";
            }

            $session['step'] = 'MAIN_MENU';
            save_session($sessionFile, $session);
            ussd_response($sessionID, $userID, $msisdn, $msg, false);
            break;

        case "0": // Exit
            $session['step'] = 'MAIN_MENU';
            save_session($sessionFile, $session);
            ussd_response($sessionID, $userID, $msisdn, "Goodbye.", false);
            break;

        default:
            ussd_response($sessionID, $userID, $msisdn, "Invalid option. Please try again.", true);
            break;
    }
}

// -------------------------
// CONTRIBUTION FLOW
// -------------------------
if($step==='CONTRIB_AMOUNT'){
    $month=date("Y-m");
    $sector=$user['sector']??'';

    // Fetch minimum
    $stmt=$conn->prepare("SELECT minimum_amount FROM sector_minimums WHERE sector_name=? LIMIT 1");
    $stmt->bind_param("s",$sector);
    $stmt->execute();
    $rowMin=$stmt->get_result()->fetch_assoc();
    $stmt->close();
    $month_min=floatval($rowMin['minimum_amount']??50);

    // Existing contribution
    $stmt=$conn->prepare("SELECT total_paid FROM monthly_contributions WHERE user_id=? AND contribution_month=? LIMIT 1");
    $uid=intval($user['id']);
    $stmt->bind_param("is",$uid,$month);
    $stmt->execute();
    $row=$stmt->get_result()->fetch_assoc();
    $stmt->close();
    $already_paid=floatval($row['total_paid']??0);
    $remaining=max(0,$month_min-$already_paid);

    if(empty($input)){
        $msg="Enter amount to contribute (GHS):\n";
        if($remaining>0) $msg.="Remaining to meet minimum: GHS ".number_format($remaining,2);
        else $msg.="You have already met this month's minimum. You can contribute extra.";
        save_session($sessionFile,$session);
        ussd_response($sessionID,$userID,$msisdn,$msg,true);
    }

    $amt=sanitize_amount($input);
    if($amt<=0){
        save_session($sessionFile,$session);
        ussd_response($sessionID,$userID,$msisdn,"Invalid amount. Enter valid GHS amount:",true);
    }

    $session['contrib_amount']=$amt;
    $session['step']='CONTRIB_PIN';
    save_session($sessionFile,$session);
    $msg="You are about to contribute GHS ".number_format($amt,2).".\n";
    if($remaining>0) $msg.="Remaining this month: GHS ".number_format($remaining,2)."\n";
    $msg.="Enter your 4-digit PIN to confirm:";
    ussd_response($sessionID,$userID,$msisdn,$msg,true);
}

if($step==='CONTRIB_PIN'){
    $pin_input=preg_replace('/\D/','',$input);
    if(!$user){
        $session['step']='GUEST_MENU';
        save_session($sessionFile,$session);
        ussd_response($sessionID,$userID,$msisdn,"User not found. Please register first.",false);
    }
    if(!password_verify($pin_input,$user['pin'])){
        $session['step']='CONTRIB_PIN';
        save_session($sessionFile,$session);
        ussd_response($sessionID,$userID,$msisdn,"Invalid PIN. Try again:",true);
    }

    $amount=floatval($session['contrib_amount']??0);
    if($amount<=0){
        $session['step']='MAIN_MENU';
        save_session($sessionFile,$session);
        ussd_response($sessionID,$userID,$msisdn,"Invalid amount. Start again.",false);
    }

    $month=date("Y-m");
    $uid=intval($user['id']);
    $sector=$user['sector']??'';

    // Fetch minimum
    $stmt=$conn->prepare("SELECT minimum_amount FROM sector_minimums WHERE sector_name=? LIMIT 1");
    $stmt->bind_param("s",$sector);
    $stmt->execute();
    $rowMin=$stmt->get_result()->fetch_assoc();
    $stmt->close();
    $month_min=floatval($rowMin['minimum_amount']??50);

    // Insert payment
    $created_at=date("Y-m-d H:i:s");
    try{$clientRef=strtoupper(substr(bin2hex(random_bytes(6)),0,12));}catch(Exception $e){$clientRef='REF'.time();}
    $stmt=$conn->prepare("INSERT INTO payments (user_id,user_phone,amount,transaction_id,status,payment_date) VALUES (?,?,?,?, 'pending', ?)");
    $stmt->bind_param("isdss",$uid,$msisdn,$amount,$clientRef,$created_at);
    $stmt->execute();
    $stmt->close();

    // Update monthly contributions
    $stmt=$conn->prepare("SELECT total_paid FROM monthly_contributions WHERE user_id=? AND contribution_month=? LIMIT 1");
    $stmt->bind_param("is",$uid,$month);
    $stmt->execute();
    $row=$stmt->get_result()->fetch_assoc();
    $stmt->close();

    $already_paid=floatval($row['total_paid']??0);
    $new_total=$already_paid+$amount;
    $status=$new_total>=$month_min?'Paid':'Pending';

    if($row){
        $stmt=$conn->prepare("UPDATE monthly_contributions SET total_paid=?, status=?, updated_at=NOW() WHERE user_id=? AND contribution_month=?");
        $stmt->bind_param("dsis",$new_total,$status,$uid,$month);
        $stmt->execute();
        $stmt->close();
    } else {
        $stmt=$conn->prepare("INSERT INTO monthly_contributions (user_id,contribution_month,total_paid,status) VALUES (?,?,?,?)");
        $stmt->bind_param("isds",$uid,$month,$new_total,$status);
        $stmt->execute();
        $stmt->close();
    }

    $session['step']='MAIN_MENU';
    save_session($sessionFile,$session);
    ussd_response($sessionID,$userID,$msisdn,"Payment of GHS ".number_format($amount,2)." recorded.\nStatus: $status",false);
}


// BENEFICIARY MAIN MENU
if ($step === 'BEN_MAIN') {
    if ($input === "1") {
        $session['step'] = 'BEN_NAME';
        save_session($sessionFile, $session);
        ussd_response($sessionID, $userID, $msisdn, "Enter beneficiary full name:", true);

    } elseif ($input === "2") {

        $stmt = $conn->prepare("SELECT name, phone, relationship FROM beneficiaries WHERE user_id=? ORDER BY added_at DESC");
        $uid = intval($user['id'] ?? 0);
        $stmt->bind_param("i", $uid);
        $stmt->execute();
        $res = $stmt->get_result();

        if ($res->num_rows === 0) {
            $stmt->close();
            ussd_response($sessionID, $userID, $msisdn, "No beneficiaries saved.", false);
        }

        $list = "Saved Beneficiaries:\n";
        while ($b = $res->fetch_assoc()) {
            $list .= trim($b['name']) . " ({$b['relationship']})\n{$b['phone']}\n\n";
        }

        $stmt->close();
        ussd_response($sessionID, $userID, $msisdn, $list, false);

    } elseif ($input === "0") {
        $session['step'] = 'MAIN_MENU';
        save_session($sessionFile, $session);
        ussd_response($sessionID, $userID, $msisdn, "Back to menu.", true);

    } else {
        ussd_response($sessionID, $userID, $msisdn, "Invalid option.\n1.Add\n2.View\n0.Back", true);
    }
}

if ($step === 'BEN_NAME') {
    $session['ben_name'] = substr(trim($input), 0, 80);
    $session['step'] = 'BEN_PHONE';
    save_session($sessionFile, $session);
    ussd_response($sessionID, $userID, $msisdn, "Enter beneficiary phone number:", true);
}

if ($step === 'BEN_PHONE') {
    $session['ben_phone'] = preg_replace('/\D+/', '', $input);
    $session['step'] = 'BEN_REL';
    save_session($sessionFile, $session);
    ussd_response($sessionID, $userID, $msisdn, "Enter relationship (e.g. Wife, Child, Mother):", true);
}
if ($step === 'BEN_REL') {
    $name = $session['ben_name'] ?? '';
    $phone = $session['ben_phone'] ?? '';
    $rel = substr(trim($input), 0, 30);

    $stmt = $conn->prepare("
        INSERT INTO beneficiaries (user_id, name, phone, relationship, added_at)
        VALUES (?, ?, ?, ?, NOW())
    ");
    $uid = intval($user['id'] ?? 0);
    $stmt->bind_param("isss", $uid, $name, $phone, $rel);

    if ($stmt->execute()) {
        $stmt->close();
        $session['step'] = 'MAIN_MENU';
        save_session($sessionFile, $session);
        ussd_response($sessionID, $userID, $msisdn, "Beneficiary saved successfully.", false);
    } else {
        $stmt->close();
        error_log("DB insert beneficiary failed: " . ($conn->error ?? ''));
        ussd_response($sessionID, $userID, $msisdn, "Failed to save beneficiary.", false);
    }
}

// SUPPORT MESSAGE FLOW
if ($step === 'SUPPORT_MESSAGE') {
    $msg_text = trim($input);

    if ($msg_text) {

        $user_id = $user['id'] ?? null;  // If user exists
        $now = date("Y-m-d H:i:s");

        $stmt = $conn->prepare("
            INSERT INTO support_messages (user_id, phone, message, date_sent) 
            VALUES (?, ?, ?, ?)
        ");
        $uid = $user_id ? intval($user_id) : null;
        // bind_param doesn't accept null for i; use 0 for anonymous
        $bind_uid = $uid ?: 0;
        $stmt->bind_param("isss", $bind_uid, $msisdn, $msg_text, $now);
        $stmt->execute();
        $stmt->close();

        // Reset back to main menu
        $session['step'] = 'MAIN_MENU';
        save_session($sessionFile, $session);

        ussd_response(
            $sessionID, 
            $userID, 
            $msisdn, 
            "Your message has been sent. Support will contact you soon.", 
            false
        );

    } else {
        ussd_response(
            $sessionID, 
            $userID, 
            $msisdn, 
            "Message cannot be empty. Enter your support message:", 
            true
        );
    }
}

// CHANGE PIN FLOW
if ($step === 'CHANGE_PIN_OLD') {
    $pin_input = preg_replace('/\D/', '', $input);
    if (!password_verify($pin_input, $user['pin'])) {
        // allow retry
        $session['step'] = 'CHANGE_PIN_OLD';
        save_session($sessionFile, $session);
        ussd_response($sessionID, $userID, $msisdn, "Invalid current PIN. Try again:", true);
    }
    $session['step'] = 'CHANGE_PIN_NEW';
    save_session($sessionFile, $session);
    ussd_response($sessionID, $userID, $msisdn, "Enter new 4-digit PIN:", true);
}

if ($step === 'CHANGE_PIN_NEW') {
    $new_pin = preg_replace('/\D/', '', $input);
    if (strlen($new_pin) !== 4) {
        ussd_response($sessionID, $userID, $msisdn, "PIN must be 4 digits. Enter new PIN:", true);
    }
    $new_pin_hash = password_hash($new_pin, PASSWORD_DEFAULT);
    $stmt = $conn->prepare("UPDATE users SET pin=? WHERE phone=?");
    $stmt->bind_param("ss", $new_pin_hash, $msisdn);
    $stmt->execute();
    $stmt->close();

    $session['step'] = 'MAIN_MENU';
    save_session($sessionFile, $session);
    ussd_response($sessionID, $userID, $msisdn, "PIN successfully changed.", false);
}

// Default catch-all: save session and show friendly error
save_session($sessionFile, $session);
ussd_response($sessionID, $userID, $msisdn, "Invalid option. Please try again.", false);
