<?php
// test_hubtel_webhook.php - Simulate Hubtel MoMo confirmation
include("config.php"); // provides $conn, send_sms()

// Simulated payload
$payload = [
    "hub_tx_id" => "HUB" . rand(100000,999999), // Simulated Hubtel transaction ID
    "client_ref" => $_GET['ref'] ?? "REF".time(), // reference sent during USSD payment
    "amount" => floatval($_GET['amount'] ?? 50),
    "phone" => preg_replace('/\D+/', '', $_GET['phone'] ?? '233201234567'),
    "status" => "success",
    "timestamp" => date("Y-m-d H:i:s")
];

// Log payload
file_put_contents("webhook_test.log", date('Y-m-d H:i:s')." RAW: ".json_encode($payload)."\n", FILE_APPEND);

// Idempotent check
$stmt = $conn->prepare("SELECT id FROM processed_webhooks WHERE hub_tx_id=? LIMIT 1");
$stmt->bind_param("s", $payload['hub_tx_id']);
$stmt->execute();
$res = $stmt->get_result();
if ($res->num_rows > 0) {
    echo "Webhook already processed\n";
    exit;
}
$stmt->close();

// Fetch the payment using client_ref
$stmt = $conn->prepare("SELECT id, user_id, user_phone, amount, status FROM payments WHERE transaction_id=? LIMIT 1");
$stmt->bind_param("s", $payload['client_ref']);
$stmt->execute();
$payment = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$payment) {
    echo "Payment not found\n";
    exit;
}

// Begin transaction
$conn->begin_transaction();
try {
    // Update payment status and hub_tx_id
    $stmt = $conn->prepare("UPDATE payments SET status='success', hub_tx_id=? WHERE id=?");
    $stmt->bind_param("si", $payload['hub_tx_id'], $payment['id']);
    $stmt->execute();
    $stmt->close();

    // Credit user balance
    $stmt = $conn->prepare("UPDATE users SET balance = balance + ? WHERE id=?");
    $stmt->bind_param("di", $payment['amount'], $payment['user_id']);
    $stmt->execute();
    $stmt->close();

    // Record webhook processed
    $stmt = $conn->prepare("INSERT INTO processed_webhooks (hub_tx_id, payment_id, date_processed) VALUES (?, ?, NOW())");
    $stmt->bind_param("si", $payload['hub_tx_id'], $payment['id']);
    $stmt->execute();
    $stmt->close();

    $conn->commit();

    // Send SMS
    $smsMsg = "Your payment of GHS ".number_format($payment['amount'],2)." has been received. Thank you!";
    send_sms($payment['user_phone'], $smsMsg);

    echo "Webhook processed successfully\n";

} catch (Exception $e) {
    $conn->rollback();
    echo "Error processing webhook: ".$e->getMessage()."\n";
}
