<?php
// ssnit_import_safe.php
// Production-ready SSNIT import with validation, duplicate check, and error logging

include("config.php"); // uses $conn from config.php
date_default_timezone_set('Africa/Accra');

// CSV file from SSNIT
$csvFile = __DIR__ . '/ssnit_update.csv';
if (!file_exists($csvFile)) {
    die("CSV file not found: $csvFile");
}

// Error log file
$errorLog = __DIR__ . '/ssnit_import_errors.log';
file_put_contents($errorLog, "SSNIT Import Errors - " . date('Y-m-d H:i:s') . PHP_EOL, FILE_APPEND);

// Optional SMS notifications
$sendSMS = true;

// Use your existing send_sms function from config.php
// function send_sms($phone, $message) {...}

// Simple SSNIT format validation (adjust regex based on SSNIT rules)
function isValidSSNIT($ssnit) {
    return preg_match('/^SSN\d{7,}$/', $ssnit); // example: SSN1234567
}

// Process CSV
if (($handle = fopen($csvFile, 'r')) !== FALSE) {
    while (($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
        $user_id = trim($data[0]);
        $new_ssnit = trim($data[1]);

        // Validate SSNIT format
        if (!isValidSSNIT($new_ssnit)) {
            file_put_contents($errorLog, "Invalid SSNIT for user_id $user_id: $new_ssnit" . PHP_EOL, FILE_APPEND);
            continue;
        }

        // Get current user info
        $stmt = $conn->prepare("SELECT ssnit_number, phone, full_name FROM users WHERE id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $stmt->bind_result($old_ssnit, $phone, $full_name);
        $stmt->fetch();
        $stmt->close();

        // Skip if user not found
        if (!$phone) {
            file_put_contents($errorLog, "User not found for user_id $user_id" . PHP_EOL, FILE_APPEND);
            continue;
        }

        // Skip if SSNIT already same as new
        if ($old_ssnit === $new_ssnit) {
            continue;
        }

        // Update users table
        $stmt = $conn->prepare("UPDATE users SET ssnit_number = ? WHERE id = ?");
        $stmt->bind_param("si", $new_ssnit, $user_id);
        if (!$stmt->execute()) {
            file_put_contents($errorLog, "Failed to update user_id $user_id: " . $stmt->error . PHP_EOL, FILE_APPEND);
            $stmt->close();
            continue;
        }
        $stmt->close();

        // Insert into audit log
        $stmt = $conn->prepare("
            INSERT INTO ssnit_update_log (user_id, old_ssnit_number, new_ssnit_number, updated_by, updated_at)
            VALUES (?, ?, ?, 'SYSTEM', ?)
        ");
        $updated_at = date("Y-m-d H:i:s");
        $stmt->bind_param("isss", $user_id, $old_ssnit, $new_ssnit, $updated_at);
        if (!$stmt->execute()) {
            file_put_contents($errorLog, "Failed to log SSNIT for user_id $user_id: " . $stmt->error . PHP_EOL, FILE_APPEND);
        }
        $stmt->close();

        // Update contribution_history table
        $stmt = $conn->prepare("UPDATE contribution_history SET ssnit_number = ? WHERE user_id = ?");
        $stmt->bind_param("si", $new_ssnit, $user_id);
        if (!$stmt->execute()) {
            file_put_contents($errorLog, "Failed to update contributions for user_id $user_id: " . $stmt->error . PHP_EOL, FILE_APPEND);
        }
        $stmt->close();

        // Send SMS notification
        if ($sendSMS) {
            $message = "Hello $full_name, your SSNIT number has been assigned: $new_ssnit";
            send_sms($phone, $message);
        }
    }
    fclose($handle);
}

$conn->close();
echo "SSNIT import completed. Check $errorLog for any errors." . PHP_EOL;
