<?php
// contribute.php
include('../config.php');
session_start();

$user_id = $_SESSION['user_id'] ?? null;
if (!$user_id) { 
    header("Location: login.php"); 
    exit; 
}

// Fetch user info
$stmt = $conn->prepare("SELECT full_name, balance, phone, sector FROM users WHERE id=? LIMIT 1");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$user = $stmt->get_result()->fetch_assoc();
$stmt->close();

// Handle form submission
$success = $error = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $amount = floatval($_POST['amount'] ?? 0);

    if ($amount <= 0) {
        $error = "Please enter a valid amount.";
    } else {
        // Fetch sector minimum
        $stmt = $conn->prepare("SELECT minimum_amount FROM sector_minimums WHERE sector_name=? LIMIT 1");
        $stmt->bind_param("s", $user['sector']);
        $stmt->execute();
        $row = $stmt->get_result()->fetch_assoc();
        $stmt->close();

        $month_min = floatval($row['minimum_amount'] ?? 50);
        if ($month_min < 1) $month_min = 50;

        // Calculate months covered
        $monthsCovered = floor($amount / $month_min);
        if ($monthsCovered < 1) $monthsCovered = 1;

        // Generate transaction reference
        $clientRef = strtoupper(substr(bin2hex(random_bytes(6)),0,12));
        $created_at = date("Y-m-d H:i:s");

        // Insert into payments table
        $stmt = $conn->prepare("INSERT INTO payments (user_id, user_phone, amount, transaction_id, status, payment_date) VALUES (?, ?, ?, ?, 'pending', ?)");
        $stmt->bind_param("isdss", $user_id, $user['phone'], $amount, $clientRef, $created_at);

        if ($stmt->execute()) {
            $stmt->close();

            // Log API request
            $req = json_encode([
                "ClientReference" => $clientRef,
                "Amount" => $amount,
                "Phone" => $user['phone'],
                "MonthsCovered" => $monthsCovered
            ], JSON_UNESCAPED_SLASHES);

            $type = 'PAYMENT_REQUEST';
            $stmt2 = $conn->prepare("INSERT INTO api_logs (response, token, date_created, type) VALUES (?, ?, ?, ?)");
            $stmt2->bind_param("ssss", $req, $clientRef, $created_at, $type);
            $stmt2->execute();
            $stmt2->close();

            // Initiate Hubtel MoMo payment
            // include('../includes/mpay_api.php'); // ensure your Hubtel API function is included
            $resp = hubtel_initiate_payment($user['phone'], $amount, $clientRef, "Contribution to EdwumaSupa");
            if (!$resp) error_log("Hubtel initiation failed for {$user['phone']}, Ref: $clientRef");

            $success = "Payment request created (Ref: $clientRef). Approve on your phone to complete.";
        } else {
            $stmt->close();
            $error = "Unable to create payment. Try again later.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Contribute - EdwumaSupa</title>
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">
<style>
body { font-family: 'Poppins', sans-serif; background: #F9F9F9; margin:0; padding:0; color:#000; }
header { background: linear-gradient(135deg, #F58220 0%, #D85C00 100%); padding:15px 20px; color:#fff; display:flex; justify-content:space-between; align-items:center; position:fixed; width:100%; top:0; z-index:1001; }
header h1 { font-size:18px; margin:0; }
.header-right { display:flex; align-items:center; gap:15px; }
.header-right .logout-link { color:#fff; font-size:18px; margin-left:5px; }
.container { padding: 90px 20px 20px 20px; max-width:400px; margin:auto; }
.balance-box { background:#fff; padding:15px; border-radius:12px; box-shadow:0 6px 20px rgba(0,0,0,0.08); text-align:center; margin-bottom:20px; }
.balance-box h3 { margin:0 0 5px; color:#D85C00; }
.balance-box span { font-size:20px; font-weight:600; }
form { background:#fff; padding:20px; border-radius:12px; box-shadow:0 6px 20px rgba(0,0,0,0.08); }
form input, form button { width:100%; padding:12px; margin-bottom:15px; border-radius:10px; border:1px solid #ccc; font-size:16px; outline:none; }
form input:focus { border-color:#F58220; box-shadow:0 0 6px rgba(245,130,32,0.3); }
form button { background: linear-gradient(135deg, #F58220, #D85C00); color:#fff; font-weight:600; border:none; cursor:pointer; transition:0.3s; }
form button:hover { background: linear-gradient(135deg, #D85C00, #F58220); }
.message { text-align:center; margin-bottom:15px; font-weight:500; }
.success { color:#28a745; }
.error { color:#dc3545; }

input {
    width: 80%;
    padding: 12px 15px;
    margin-bottom: 12px;
    border-radius: 8px;
    border: 1px solid #ccc;
    font-size: 16px;
    box-sizing: border-box;
}
</style>
</head>
<body>

<header>
    <div class="back" onclick="window.location='index.php'"><i class="fa-solid fa-arrow-left"></i></div>
    <div class="header-right">
        <span><i class="fa-solid fa-user"></i> <?=htmlspecialchars($user['full_name'])?></span>
        <a href="logout.php" class="logout-link"><i class="fa-solid fa-right-from-bracket"></i></a>
    </div>
</header>

<div class="container">

    <div class="balance-box">
        <h3>Current Balance</h3>
        <span>₵ <?=number_format($user['balance'],2)?></span>
    </div>

    <?php if($success): ?>
        <div class="message success"><?=htmlspecialchars($success)?></div>
    <?php endif; ?>
    <?php if($error): ?>
        <div class="message error"><?=htmlspecialchars($error)?></div>
    <?php endif; ?>

    <form method="POST" action="">
        <input type="number" step="0.01" name="amount" placeholder="Enter contribution amount (₵)" required>
        <button type="submit">Contribute</button>
    </form>

</div>
<footer>
    &copy; <?=date('Y')?> 3y3 Supa. Powered by Tekpulse Consult Ltd.
</footer>

<style>
footer {
    position: fixed;
    bottom: 0;
    left: 0;
    width: 100%;
    background: #f5f5f5; /* light grey / whitish */
    color: #333; /* dark text for contrast */
    text-align: center;
    padding: 12px 0;
    font-size: 14px;
    box-shadow: 0 -3px 10px rgba(0,0,0,0.1);
}

footer a {
    color: #002147; /* optional: make any links dark blue */
    text-decoration: none;
}

@media (max-width: 400px) {
    footer {
        font-size: 12px;
        padding: 10px 0;
    }
}
</style>


</body>
</html>
