<?php
include("config.php");
date_default_timezone_set('Africa/Accra');

// --- Read and decode raw POST JSON ---
$raw = file_get_contents('php://input');
$data = json_decode($raw, true);

// Validate JSON input
if (!$data) {
    http_response_code(400);
    exit(json_encode(["status"=>"error","message"=>"Invalid JSON payload"]));
}

// --- Extract callback fields ---
$reference = $data['ClientReference'] ?? $data['Reference'] ?? $data['TransactionId'] ?? null;
$status    = strtoupper($data['Status'] ?? '');
$amount    = floatval($data['Amount'] ?? 0);
$phone     = preg_replace('/\D/', '', $data['PhoneNumber'] ?? $data['CustomerPhoneNumber'] ?? '');
$extTx     = $data['ExternalTransactionId'] ?? $data['TransactionId'] ?? null;
$received  = date("Y-m-d H:i:s");

// --- Validate reference ---
if (!$reference) {
    $stmt = $conn->prepare("INSERT INTO api_logs (response, token, date_created, type) VALUES (?, ?, ?, 'CALLBACK_NO_REFERENCE')");
    $stmt->bind_param("sss", $raw, $reference, $received);
    $stmt->execute();
    $stmt->close();
    http_response_code(400);
    exit(json_encode(["status"=>"error","message"=>"Missing client reference"]));
}

// --- Normalize phone to 233XXXXXXXXX ---
if (strlen($phone) === 10 && $phone[0] === '0') $phone = "233".substr($phone,1);

// --- Fetch payment record ---
$stmt = $conn->prepare("SELECT id, user_id, status, amount, payment_date FROM payments WHERE transaction_id=? LIMIT 1");
$stmt->bind_param("s", $reference);
$stmt->execute();
$payment = $stmt->get_result()->fetch_assoc();
$stmt->close();

// --- Unknown transaction? log & exit ---
if (!$payment) {
    $stmt = $conn->prepare("INSERT INTO api_logs (response, token, date_created, type) VALUES (?, ?, ?, 'CALLBACK_UNKNOWN')");
    $stmt->bind_param("sss", $raw, $reference, $received);
    $stmt->execute();
    $stmt->close();
    http_response_code(404);
    exit(json_encode(["status"=>"error","message"=>"Unknown payment reference"]));
}

// --- Prevent double processing & verify amount ---
if ($payment['status'] === 'success') {
    http_response_code(200);
    exit(json_encode(["status"=>"ok","message"=>"Already processed"]));
}

if (floatval($payment['amount']) != $amount) {
    http_response_code(400);
    exit(json_encode(["status"=>"error","message"=>"Amount mismatch"]));
}

// --- Process successful payment ---
if (in_array($status, ['SUCCESS','COMPLETED','APPROVED','0000'])) {
    $conn->begin_transaction();
    try {
        // 1️⃣ Update payments table
        $stmt = $conn->prepare("UPDATE payments SET status='success', payment_date=?, external_transaction_id=? WHERE transaction_id=?");
        $stmt->bind_param("sss", $received, $extTx, $reference);
        $stmt->execute();
        $stmt->close();

        // 2️⃣ Update user balance
        $stmt = $conn->prepare("UPDATE users SET balance = balance + ? WHERE id=?");
        $stmt->bind_param("di", $amount, $payment['user_id']);
        $stmt->execute();
        $stmt->close();

        // 3️⃣ Allocate to monthly_contributions
        $stmt = $conn->prepare("SELECT s.minimum_amount FROM sector_minimums s JOIN users u ON u.sector = s.sector_name WHERE u.id=? LIMIT 1");
        $stmt->bind_param("i", $payment['user_id']);
        $stmt->execute();
        $sectorData = $stmt->get_result()->fetch_assoc();
        $stmt->close();

        $month_min = floatval($sectorData['minimum_amount'] ?? 50);
        if ($month_min < 1) $month_min = 50;

        $monthsCovered = floor($amount / $month_min);
        if ($monthsCovered < 1) $monthsCovered = 1;

        $remainingAmount = $amount;
        $currentMonth = date("Y-m");

        for ($i = 0; $i < $monthsCovered; $i++) {
            // Check existing contribution
            $stmt = $conn->prepare("SELECT total_paid FROM monthly_contributions WHERE user_id=? AND contribution_month=? LIMIT 1");
            $stmt->bind_param("is", $payment['user_id'], $currentMonth);
            $stmt->execute();
            $row = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            $toAdd = ($remainingAmount >= $month_min) ? $month_min : $remainingAmount;
            $statusMonth = ($row ? $row['total_paid'] + $toAdd : $toAdd) >= $month_min ? 'PAID' : 'PENDING';

            if ($row) {
                $stmt = $conn->prepare("UPDATE monthly_contributions SET total_paid=total_paid+?, status=? WHERE user_id=? AND contribution_month=?");
                $stmt->bind_param("dsis", $toAdd, $statusMonth, $payment['user_id'], $currentMonth);
            } else {
                $stmt = $conn->prepare("INSERT INTO monthly_contributions (user_id, contribution_month, total_paid, status) VALUES (?, ?, ?, ?)");
                $stmt->bind_param("isds", $payment['user_id'], $currentMonth, $toAdd, $statusMonth);
            }
            $stmt->execute();
            $stmt->close();

            $remainingAmount -= $toAdd;
            $currentMonth = date("Y-m", strtotime("$currentMonth +1 month"));
            if ($remainingAmount <= 0) break;
        }

        // 4️⃣ Log successful callback
        $stmt = $conn->prepare("INSERT INTO api_logs (response, token, date_created, type) VALUES (?, ?, ?, 'CALLBACK_SUCCESS')");
        $stmt->bind_param("sss", $raw, $reference, $received);
        $stmt->execute();
        $stmt->close();

        $conn->commit();
        http_response_code(200);
        echo json_encode(["status"=>"OK","message"=>"Payment processed successfully"]);
        exit;

    } catch (Exception $e) {
        $conn->rollback();
        error_log("Callback error: ".$e->getMessage());
        http_response_code(500);
        echo json_encode(["status"=>"error","message"=>"Internal server error"]);
        exit;
    }
} else {
    // Failed payment
    $logType = ($payment['status'] === 'success') ? 'CALLBACK_IGNORED' : 'CALLBACK_FAILED';
    $stmt = $conn->prepare("INSERT INTO api_logs (response, token, date_created, type) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("ssss", $raw, $reference, $received, $logType);
    $stmt->execute();
    $stmt->close();

    http_response_code(200);
    echo json_encode(["status"=>"ignored","message"=>"Payment failed or ignored"]);
    exit;
}
?>
