<?php
session_start();
require 'config.php';

// ==================== VALIDATE SESSION ====================
if (!isset($_SESSION['agent_name']) || !isset($_SESSION['agent_id'])) {
    header("Location: login.php");
    exit();
}

$agent_id = $_SESSION['agent_id'];

// ==================== GET POST DATA ====================
$user_id = intval($_POST['user_id'] ?? 0);
$amount = floatval($_POST['amount'] ?? 0);
$notes = trim($_POST['notes'] ?? '');

// ==================== BASIC VALIDATION ====================
if ($user_id <= 0 || $amount <= 0) {
    $_SESSION['error'] = "Invalid contributor or amount.";
    header("Location: collect_payment.php");
    exit();
}

// ==================== CHECK CONTRIBUTOR EXISTS ====================
$stmt = $conn->prepare("SELECT full_name, phone, balance, sector FROM users WHERE id = ? AND status = 'active'");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    $_SESSION['error'] = "Contributor not found or inactive.";
    header("Location: collect_payment.php");
    exit();
}

$user = $result->fetch_assoc();
$current_balance = floatval($user['balance']);
$user_phone = $user['phone'];
$sector = $user['sector'] ?? '';
$month = date("Y-m");

// ==================== START TRANSACTION ====================
$conn->begin_transaction();

try {
    // 1️⃣ Generate a transaction ID
    $transaction_id = 'TX' . time() . rand(100, 999);

    // 2️⃣ Insert payment
    $stmt = $conn->prepare("
        INSERT INTO payments 
        (user_id, agent_id, user_phone, amount, transaction_id, status, payment_date)
        VALUES (?, ?, ?, ?, ?, 'success', NOW())
    ");
    $stmt->bind_param("iisss", $user_id, $agent_id, $user_phone, $amount, $transaction_id);
    $stmt->execute();

    // 3️⃣ Update user's balance
    $new_balance = $current_balance + $amount;
    $stmt = $conn->prepare("UPDATE users SET balance = ? WHERE id = ?");
    $stmt->bind_param("di", $new_balance, $user_id);
    $stmt->execute();

    // 4️⃣ Update monthly contributions
    // Fetch sector minimum
    $stmt = $conn->prepare("SELECT minimum_amount FROM sector_minimums WHERE sector_name=? LIMIT 1");
    $stmt->bind_param("s", $sector);
    $stmt->execute();
    $rowMin = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    $month_min = floatval($rowMin['minimum_amount'] ?? 50);

    // Fetch existing contribution
    $stmt = $conn->prepare("SELECT total_paid FROM monthly_contributions WHERE user_id=? AND contribution_month=? LIMIT 1");
    $stmt->bind_param("is",$user_id,$month);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    $already_paid = floatval($row['total_paid'] ?? 0);
    $new_total = $already_paid + $amount;
    $status = $new_total >= $month_min ? 'Paid' : 'Pending';

    if ($row) {
        $stmt = $conn->prepare("UPDATE monthly_contributions SET total_paid=?, status=?, updated_at=NOW() WHERE user_id=? AND contribution_month=?");
        $stmt->bind_param("dsis",$new_total,$status,$user_id,$month);
        $stmt->execute();
        $stmt->close();
    } else {
        $stmt = $conn->prepare("INSERT INTO monthly_contributions (user_id, contribution_month, total_paid, status) VALUES (?,?,?,?)");
        $stmt->bind_param("isds",$user_id,$month,$new_total,$status);
        $stmt->execute();
        $stmt->close();
    }

    // 5️⃣ Commit transaction
    $conn->commit();

    $_SESSION['success'] = "Payment of ₵$amount recorded for " . $user['full_name'] . ". Status this month: $status";
    header("Location: collect_payment.php");
    exit();

} catch (Exception $e) {
    $conn->rollback();
    $_SESSION['error'] = "Payment failed: " . $e->getMessage();
    header("Location: collect_payment.php");
    exit();
}
