<?php
session_start();
require_once 'includes/db_connect.php';

// Security check
if (!isset($_SESSION['agent_name'])) {
    header("Location: login.php");
    exit();
}

$agent_name = $_SESSION['agent_name'];

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header("Location: agent_onboard.php");
    exit();
}

/* ---------------------------
   Sanitize Inputs
--------------------------- */
function clean($data) {
    return htmlspecialchars(trim($data));
}

$full_name       = clean($_POST['full_name'] ?? '');
$phone           = clean($_POST['phone'] ?? '');
$gender          = clean($_POST['gender'] ?? '');
$dob             = $_POST['dob'] ?? null;
$id_type         = clean($_POST['id_type'] ?? '');
$id_number       = clean($_POST['id_number'] ?? '');
$location        = clean($_POST['location'] ?? '');
$ssnit_number    = clean($_POST['ssnit_number'] ?? '');
$initial_amount  = $_POST['initial_amount'] ?? 0;

/* ---------------------------
   Validation
--------------------------- */
if (empty($full_name) || empty($phone) || empty($gender) || empty($dob)) {
    $_SESSION['error'] = "Please fill all required fields.";
    header("Location: agent_onboard.php");
    exit();
}

// Phone basic validation (Ghana)
if (!preg_match('/^(0|\+233)[0-9]{9}$/', $phone)) {
    $_SESSION['error'] = "Invalid phone number format.";
    header("Location: agent_onboard.php");
    exit();
}

// Prevent duplicate phone
$check = $conn->prepare("SELECT id FROM contributors WHERE phone = ?");
$check->bind_param("s", $phone);
$check->execute();
$check->store_result();

if ($check->num_rows > 0) {
    $_SESSION['error'] = "Contributor with this phone number already exists.";
    header("Location: agent_onboard.php");
    exit();
}
$check->close();

/* ---------------------------
   Insert Contributor
--------------------------- */
$stmt = $conn->prepare("
    INSERT INTO contributors
    (full_name, phone, gender, dob, id_type, id_number, location, ssnit_number, initial_amount, agent_name, created_at)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
");

$stmt->bind_param(
    "ssssssssds",
    $full_name,
    $phone,
    $gender,
    $dob,
    $id_type,
    $id_number,
    $location,
    $ssnit_number,
    $initial_amount,
    $agent_name
);

if ($stmt->execute()) {

    $contributor_id = $stmt->insert_id;

    // OPTIONAL: log initial contribution
    if ($initial_amount > 0) {
        $pay = $conn->prepare("
            INSERT INTO contributions (contributor_id, amount, agent_name, created_at)
            VALUES (?, ?, ?, NOW())
        ");
        $pay->bind_param("ids", $contributor_id, $initial_amount, $agent_name);
        $pay->execute();
        $pay->close();
    }

    $_SESSION['success'] = "Contributor onboarded successfully.";
    header("Location: agent_dashboard.php");
    exit();

} else {
    $_SESSION['error'] = "Failed to save contributor. Please try again.";
    header("Location: agent_onboard.php");
    exit();
}

$stmt->close();
$conn->close();
