<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once "config.php";

if ($_SERVER['REQUEST_METHOD'] !== 'POST' || !isset($_FILES['csv_file'])) {
    echo "No file uploaded.";
    exit;
}

$send_sms_flag = isset($_POST['send_sms']) && $_POST['send_sms'] == '1';

$file = $_FILES['csv_file'];
if ($file['error'] !== UPLOAD_ERR_OK) {
    echo "Upload error: " . $file['error'];
    exit;
}
if ($file['size'] > 10 * 1024 * 1024) {
    echo "File too large (max 10MB).";
    exit;
}

// Minimal CSV expected headers
$expected_headers = ["full_name","phone","email","network","region","district","community"];

// Move uploaded file
$upload_dir = __DIR__ . "/uploads";
if (!is_dir($upload_dir)) mkdir($upload_dir, 0755, true);
$uploaded_path = $upload_dir . "/" . uniqid("bulk_", true) . ".csv";
if (!move_uploaded_file($file['tmp_name'], $uploaded_path)) {
    echo "Failed to save uploaded file.";
    exit;
}

// Open CSV
$handle = fopen($uploaded_path, "r");
if ($handle === false) {
    echo "Failed to open uploaded CSV.";
    exit;
}

// Read header
$header = fgetcsv($handle);
if ($header === false) {
    echo "Empty CSV file.";
    exit;
}

// Trim spaces and lowercase for comparison
$header_lc = array_map(function($h){ return strtolower(trim($h)); }, $header);

// Check if all expected headers exist
$missing = array_diff($expected_headers, $header_lc);
if (!empty($missing)) {
    fclose($handle);
    echo "Invalid CSV header. Missing columns: " . implode(", ", $missing);
    exit;
}

// Get CSV column indexes in expected order
$col_indexes = [];
foreach ($expected_headers as $col) {
    $col_indexes[] = array_search($col, $header_lc);
}

// Prepare DB insert (minimal columns + pin)
$insert_cols = $expected_headers;
$placeholders = implode(',', array_fill(0, count($insert_cols), '?'));
$col_list = implode(',', $insert_cols);
$sql = "INSERT INTO users ($col_list, pin, date_registered) VALUES ($placeholders, ?, NOW())";
$stmt = $conn->prepare($sql);
if (!$stmt) {
    fclose($handle);
    echo "DB prepare error: " . $conn->error;
    exit;
}

// Normalize Ghana phone
function normalize_ghana_phone($phone) {
    $phone = preg_replace('/[^0-9]/', '', $phone);
    if (strlen($phone) == 10 && substr($phone,0,1)=='0') return '233'.substr($phone,1);
    if (strlen($phone) == 9) return '233'.$phone;
    if (strlen($phone) == 12 && substr($phone,0,3)=='233') return $phone;
    return false;
}

// Generate 4-digit PIN
function generate_pin() {
    return str_pad(rand(0,9999),4,'0',STR_PAD_LEFT);
}

$total = $inserted = $skipped = 0;
$errors = [];
$phones_seen = [];

$conn->begin_transaction();
$rowNumber = 1;

while (($row = fgetcsv($handle)) !== false) {
    $rowNumber++;
    $total++;

    // Map row to expected columns
    $row_data = [];
    foreach ($col_indexes as $i) {
        $row_data[] = trim($row[$i] ?? '');
    }
    $data = array_combine($expected_headers, $row_data);

    $fail_reasons = [];
    if (empty($data['full_name'])) $fail_reasons[] = "full_name missing";
    if (empty($data['phone'])) $fail_reasons[] = "phone missing";
    if (empty($data['network'])) $fail_reasons[] = "network missing";

    $norm_phone = normalize_ghana_phone($data['phone']);
    if ($norm_phone === false) {
        $fail_reasons[] = "invalid Ghana phone number";
    } else {
        $data['phone'] = $norm_phone;
    }

    // Check duplicates in CSV
    if (isset($phones_seen[$data['phone']])) $fail_reasons[] = "duplicate phone in CSV (row {$phones_seen[$data['phone']]})";
    else $phones_seen[$data['phone']] = $rowNumber;

    // Check existing in DB
    $q = $conn->prepare("SELECT id FROM users WHERE phone=? LIMIT 1");
    $q->bind_param("s", $data['phone']);
    $q->execute();
    $q->store_result();
    if ($q->num_rows>0) $fail_reasons[] = "phone already exists in system";
    $q->close();

    if (!empty($fail_reasons)) {
        $skipped++;
        $errors[] = ['row'=>$rowNumber,'reason'=>implode("; ",$fail_reasons),'data'=>$data];
        continue;
    }

    // Generate PIN
    $pin_plain = generate_pin();
    $pin_hashed = password_hash($pin_plain,PASSWORD_DEFAULT);

    // Bind params
    $types = str_repeat('s', count($insert_cols)) . 's'; // add PIN
    $values = [];
    foreach ($insert_cols as $col) $values[] = $data[$col] ?? null;
    $values[] = $pin_hashed;

    $bind_names[] = $types;
    for ($i=0;$i<count($values);$i++){
        $bind_name='bind'.$i;
        $$bind_name = $values[$i];
        $bind_names[] = &$$bind_name;
    }
    call_user_func_array([$stmt,'bind_param'],$bind_names);

    if (!$stmt->execute()) {
        $skipped++;
        $errors[] = ['row'=>$rowNumber,'reason'=>"DB insert error: ".$stmt->error,'data'=>$data];
        $bind_names=[];
        continue;
    } else {
        $inserted++;
        if ($send_sms_flag) {
            $loginLink="https://tuliyapayinc.com/edwumasupa/mobile/login.php";
            $fullname=$data['full_name'];
            $phone=$data['phone'];
            $smsMsg="Welcome $fullname to 3y3 Supa!\nLogin with phone: $phone and PIN: $pin_plain\nPortal: $loginLink";
            send_sms($phone,$smsMsg);
        }
    }

    $bind_names=[];
}

$conn->commit();
$stmt->close();
fclose($handle);

// Show summary
echo "<h3>Bulk Upload Result</h3>";
echo "<p>Total rows processed: {$total}</p>";
echo "<p>Inserted: {$inserted}</p>";
echo "<p>Skipped: {$skipped}</p>";
if(!empty($errors)){
    echo "<p>Example reasons:</p><ul>";
    foreach(array_slice($errors,0,10) as $e){
        echo "<li>Row {$e['row']}: {$e['reason']}</li>";
    }
    echo "</ul>";
}
echo "<p><a href='bulk_upload.php'>Back</a></p>";
?>
